//********************************************************************************
//
//                       C O P Y R I G H T  (c) 2011
//                S O F T W A R E   T O O L B O X   I N C.
//                           All Rights Reserved.
//   ........................................................................
//   This sample code is provided by Software Toolbox solely to sist in
//   understanding the use of the SLIK OPC Data Access ActiveX Control. This
//   code is provided -is and without warranty or support of any sort.
//
//********************************************************************************
//
//   Project:		SLIK-DA Interop Assembly
//
//   Description:   This sample server application is based on Software Toolbox's
//                  Simple Language Independent Toolkit for creating OPC Data
//                  Access servers (SLIK-DA).  It leverages the SLIK-DA Interop
//					Assembly that allows for the easy integration of this
//					ActiveX control into .NET applications.
//
//					This implementation replicates the VB Advanced Server originally
//					created to demonstrate various features of the SLIK-DA ActiveX 
//					Control in the VB6 environment.
//
//********************************************************************************

using System.Windows.Forms;
using System.Runtime.InteropServices;
using NDI.SLIKDA.Interop;
using CSAdvancedServer;


namespace CSAdvancedServer
{
	//
	//	The CSimulate class provides services that were originally provided
	//	in the "mdlSimulate" Module in the VB Advanced Server sample.
	//

	public class CSimulate
	{	
		public static void UpdateTags( 
			ISLIKTags Tags 
			)
		{
			//
			// A real server might scan one or more devices here to update the Tags.
			// This server does not communicate with any hardware, so instead we will
			// simulate data changes.
			//
   
			// Update each SLIKTag in the SLIKTags collection.  Note that SLIKTags
			// collection indices are 1-based!
			for( int i = 1; i <= Tags.Count; i++ )
			{
				//
				// Simulate data changes for each tag that meets the following criteria:
				//   1. The tag is currently active.
				//   2. The tag's access permessions are read-only.
				//
				ISLIKTag Tag = Tags[i];
				if ( Tag.Active && ( Tag.AccessPermissions == (int) AccessPermissionsEnum.sdaReadAccess ) )
					SimulateDataChange( Tag, (int) QualityStatusEnum.sdaGood );
				else
					// Update the quality (and timestamp) for the other tags
					Tag.SetVQT( DefaultValues.SetVQT_Value, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.SetVQT_Timestamp 
						);
			}
		}

		public static void ReadTags( 
			int				Count,
			ISLIKTag[]		Tags,
			int[]			Errors,
			ref int			Result
			)
		{
			//
			//	A real server might have to read one or more devices here to service the
			//	Read request.  This server does not communicate with any hardware.  
			//	Instead, we rely on the periodic tag update to simulate data changes, so
			//	this Read call will be basically a no-op.
			//
	    
			// Update the tag level error codes.
			for ( int i = 0; i < Count; i++ )
			{
				// For this server, Tag reads always succeed!
				Errors[i] = (int) OPCDAErrorsEnum.sdaSOK;
			}    
	    
			// Return overall status.  For this server, the overall Read status is
			// always sdaSOK.  A real server should set this according to the tag level
			// read results.
			Result = (int) OPCDAErrorsEnum.sdaSOK;
		}

		public static void WriteTags( 
			int				Count,
			ISLIKTag[]		Tags,
			object[]		Values,
			short[]			Qualities,
			System.DateTime[] Timestamps,
			int[]			Errors,
			ref int			Result
			)
		{
			// Our simulation server does not talk to any hardware, so we just proceed to
			// write the new values provided to the corresponding Tag.  A real server may
			// validate the data first (e.g. range checking), scale the data, etc. before
			//performing the actual write to the underlying system/hardware.

			short Quality = (short) QualityStatusEnum.sdaGood;
			System.DateTime Timestamp = DefaultValues.SetVQT_Timestamp;

			// Write each tag
			for( int i = 0; i < Count; i++ )
			{
				try
				{
					object Val;
					// Handle the special case of VT_CY values
					if( Values[i].GetType() == typeof( System.Decimal ) )
						Val = new CurrencyWrapper( Values[i] );
					else
						Val = Values[i];

					if ( Qualities.Length > 0 )
						Quality = Qualities[i];
					if ( Timestamps.Length > 0 )
						Timestamp = Timestamps[i];

					Tags[i].SetVQT( 
						Val,
						Quality, 
						Timestamp
						);
					Errors[i] = (int) OPCDAErrorsEnum.sdaSOK;
				}
				catch( System.Exception )
				{
					Errors[i] = (int) OPCDAErrorsEnum.sdaEFail;
				}
			}
	    
			// Return overall status.  For this server, the overall Write status is
			// always sdaSOK.  A real server should set this according to the tag level
			// write results.
			Result = (int) OPCDAErrorsEnum.sdaSOK;
		}


		private static void SimulateDataChange(
			ISLIKTag	Tag,
			int			nQuality
			)
		{    
			object Value = Tag.Value;
			switch( (VarEnum) Tag.DataType )
			{
				case VarEnum.VT_BOOL:
					bool bVal = (bool) Value;
					Value = !bVal;
					break;

				case VarEnum.VT_UI1:
					byte byVal = (byte) Value;
					if( byVal > 250 )
						byVal = 0;
					else
						++byVal;
					Value = byVal;
					break;

				case VarEnum.VT_I2:
					short shVal = (short) Value;
					if( shVal > 10000 )
						shVal = -10000;
					else
						shVal += 4;
					Value = shVal;
					break;

				case VarEnum.VT_I4:
					int nVal = (int) Value;
					if( nVal > 1000000 ) 
						nVal = -1000000;
					else
						nVal += 9;
					Value = nVal;
					break;

				case VarEnum.VT_R4:
					float fltVal = (float) Value;
					if( fltVal > 1E+20 )
						fltVal = -1E+20f;
					if( fltVal > 0 )
						fltVal *= 1.1f;
					else
						fltVal = (fltVal / 1.1f) + 1f;
					Value = fltVal;
					break;

				case VarEnum.VT_R8:
					double dblVal = (double) Value;
					if( dblVal > 1E+50 )
						dblVal = -1E+50;
					if( dblVal > 0 )
						dblVal *= 1.2F;
					else
						dblVal = (dblVal / 1.2F) + 1F;
					Value = dblVal;
					break;

				case VarEnum.VT_CY:
					Value = new CurrencyWrapper( (decimal) Value + 1 );
					break;

				case VarEnum.VT_DATE:
					Value = System.DateTime.Now;
					break;

				case VarEnum.VT_BSTR:
					Value = "Current time is " + System.DateTime.Now.ToLongTimeString();
					break;

				default:
					// Simulation for data type not supported
					break;
			}

			// Now set the Value, Quality, and Timestamp
			// Note: By passing the default value for the timestamp to SetVQT(),
			//       SLIK-DA will use the current system time,
			//       and generate a timestamp with millisecond resolution.
			Tag.SetVQT( 
				Value, 
				(short) nQuality, 
				DefaultValues.SetVQT_Timestamp 
				);
		}
	}
}	
